function [output_areas ] = LUCM3_save_results(output_path, share, areas, Co2price, total_endog_area )
%
% Land Use Change Module - function 3 (LUCM3)
%
% This file saves the estimations from this module and some associated
% figures.
%

% Code by Simon Anastasiadis : 2011-11-01

%% Parameters

% module output folder
module_folder = '\Results 1 Land Use Change Module';
% figures output folder
figure_folder = '\Results 1 Land Use Change Module\figures';

% number of hectares of rural land in LURNZ
LURNZ_hectares = total_endog_area;

%% Set folder to store results

% make module folder
if ~exist([output_path,module_folder],'dir')
    mkdir([output_path,module_folder]);
end

% make figures folder
if ~exist([output_path,figure_folder],'dir')
    mkdir([output_path,figure_folder]);
end

%% convert areas to shares
% share is a matrix
% areas is a Matlab structure

% convert areas to matrix
areas = [areas.year, areas.dairy, areas.sheep, areas.forest, areas.scrub];

% convert areas to percentages
areas(:,2:5) = areas(:,2:5) ./ (sum(areas(:,2:5),2) * ones(1,4));
% scale by other land
areas(:,2:5) = areas(:,2:5) * sum(share(1,2:5));


%% Combine shares and areas

% initialize
min_year = min(areas(:,1));
max_year = max(share(:,1));
output_shares = zeros(max_year - min_year + 1,size(share,2));

% years
output_shares(:,1) = (min_year:max_year).';

% fill from areas
output_shares(1:size(areas,1),2:5) = areas(:,2:5);
% fill from shares
start_yr = find(output_shares(:,1)==share(1,1));
end_yr   = find(output_shares(:,1)==share(end,1));
output_shares(start_yr:end_yr,2:5) = share(:,2:5);

%% Scale output_shares for LURNZ
% currently output shares are scaled to match Olssen and Kerr (2011) data.

output_shares(:,2:5) = output_shares(:,2:5) / sum(output_shares(1,2:5));

%% Land Use Areas

% initialize
output_areas = output_shares;
% scale by total endogenous land area
output_areas(:,2:5) = output_areas(:,2:5) * LURNZ_hectares;

%% Land Use Change

% take differences (change_t = {t} - {t-1} )
output_change = output_areas(2:end,:) - output_areas(1:end-1,:);
% year values (year_t = {t} )
output_change(:,1) = output_areas(2:end,1);

%% Excel output - reused titles

% sheet name
sheet = ['Co2 price = $',num2str(Co2price)];
% sheet = 'Sheet1';
% file appendix
file_appendix = [num2str(share(1,1)),'_to_',num2str(share(end,1)),'.xls'];

%% Disable Warnings for adding Excel sheets

warning('OFF','MATLAB:xlswrite:AddSheet')

%% Excel output: Land Use Share

% Column header
column_header = {'Year','Dairy share of endogenous land', ...
                 'Sheep & Beef share of endogenous land', ...
                       'Forest share of endogenous land', ...
                        'Scrub share of endogenous land', ...
                        strcat('Co2price = $',num2str(Co2price)) };
% file name
file_name = [output_path,module_folder,'\LU_Share_',file_appendix];

% save output
xlswrite(file_name, column_header, sheet,'A1');
xlswrite(file_name, output_shares, sheet,'A2');

%% Excel output: Land Use Areas

% Column header
column_header = {'Year','Dairy area (ha)', ...
                 'Sheep & Beef area (ha)', ...
                       'Forest area (ha)', ...
                        'Scrub area (ha)', ...
                        strcat('Co2price = $',num2str(Co2price)) };
% file name
file_name = [output_path,module_folder,'\LU_Area_',file_appendix];

% save output
xlswrite(file_name, column_header, sheet,'A1');
xlswrite(file_name, output_areas, sheet,'A2');

%% Excel output: Land Use Change

% Column header
column_header = {'Year','Dairy yearly area change (ha)', ...
                 'Sheep & Beef yearly area change (ha)', ...
                       'Forest yearly area change (ha)', ...
                        'Scrub yearly area change (ha)', ...
                        strcat('Co2price = $',num2str(Co2price)) };
% file name
file_name = [output_path,module_folder,'\LU_Change_',file_appendix];

% save output
xlswrite(file_name, column_header, sheet,'A1');
xlswrite(file_name, output_change, sheet,'A2');

%% Enable Warnings for adding Excel sheets

warning('ON','MATLAB:xlswrite:AddSheet')

%% Graphic output: Land Use Share

% title and axis label
fig_title = ['Co2 price is $',num2str(Co2price)];
fig_y = 'Land use share (percent of endogenous rural land)';
% file name
file_name = [output_path,figure_folder,'\LU_share_Co2$',num2str(Co2price),'.png'];

% plot figure
plot_fig(output_shares, share(:,1), fig_y, fig_title, file_name, 1)

%% Graphic output: Land Use Areas

% title and axis label
fig_title = ['Co2 price is $',num2str(Co2price)];
fig_y = 'Land use areas (million hectares)';
% file name
file_name = [output_path,figure_folder,'\LU_areas_Co2$',num2str(Co2price),'.png'];

% plot figure
plot_fig([output_areas(:,1),output_areas(:,2:5)/1e6], share(:,1), fig_y, fig_title, file_name, 10)

%% Matlab output

% clear excess variables
clear('start_yr', 'sheet', 'share', 'min_year', 'max_year', 'file_name',...
      'file_appendix', 'figure_folder', 'fig_y', 'fig_title', 'end_yr',...
      'column_header', 'areas');
% save output
save([output_path,module_folder,'\LUCM_output_data'])

%% Inform User saving land use change is complete

msg = sprintf(' - - LUCM3 : save data complete');
disp(msg)

end


%% Subfunction : Remove Blank Excel sheets

function plot_fig(fig_data, years, fig_y, fig_title, file_name, scale)

% plot figure
plot(fig_data(:,1),fig_data(:,2:5),'LineWidth',2);
% constrain axis
axis([fig_data(1,1) fig_data(end,1) 0 scale]);
% title and axis label
title(fig_title);
ylabel(fig_y);
% dividing line between predictions and historical
line([years(1)+0.5; years(1)+0.5],[0;scale],'LineStyle','-','Color','k','LineWidth',1);
text(years(1)+0.5,scale/2,'\rightarrow Predictions','FontSize',12);
text(years(1)+0.5,scale/2,'Historical data \leftarrow','HorizontalAlignment','right','FontSize',12);
% legend
legend('Dairy','Sheepbeef' ,'Plantation', 'Scrub','Location','Best',6);

% save figure and close
saveas(gcf,file_name);
close;

end

